<?php

/**
 * Shortcodes.
 *
 * @since 1.0.0
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('LTY_Shortcodes')) {

	/**
	 * Class.
	 *
	 * @since 1.0.0
	 */
	class LTY_Shortcodes {

		/**
		 * Class.
		 *
		 * @since 1.0.0
		 * */
		public static function init() {
			/**
			 * This hook is used to alter the load shortcodes.
			 *
			 * @since 1.0.0
			 */
			$shortcodes = apply_filters(
					'lty_load_shortcodes',
					array(
						'lty_dashboard',
						'lty_lottery_products_winners_list',
						'lty_lottery_winners_by_date',
						'lty_lottery_instant_winners_by_date',
						'lty_my_lottery_products',
						'lty_all_lottery_products',
						'lty_ongoing_lottery_products',
						'lty_ending_soon_lottery_products',
						'lty_future_lottery_products',
						'lty_featured_lottery_products',
						'lty_closed_lottery_products',
						'lty_finished_lottery_products',
						'lty_random_lottery_products',
					)
			);

			foreach ($shortcodes as $shortcode_name) {
				add_shortcode($shortcode_name, array( __CLASS__, 'process_shortcode' ));
			}
		}

		/**
		 * Process Shortcode.
		 *
		 * @since 1.0.0
		 * @param array  $atts Shortcode attributes.
		 * @param string $content
		 * @param string $tag
		 * @return string
		 */
		public static function process_shortcode( $atts, $content, $tag ) {
			$shortcode_name = str_replace('lty_', '', $tag);
			$function = 'shortcode_' . $shortcode_name;

			switch ($shortcode_name) {
				case 'dashboard':
				case 'lottery_products_winners_list':
				case 'lottery_winners_by_date':
				case 'lottery_instant_winners_by_date':
					ob_start();
					self::$function($atts, $content);
					$content = ob_get_contents();
					ob_end_clean();
					break;

				case 'my_lottery_products':
				case 'all_lottery_products':
				case 'ongoing_lottery_products':
				case 'ending_soon_lottery_products':
				case 'future_lottery_products':
				case 'featured_lottery_products':
				case 'closed_lottery_products':
				case 'finished_lottery_products':
				case 'random_lottery_products':
					$content = self::$function($atts, $content);
					break;
				default:
					ob_start();

					/**
					 * This hook is used to display the shortcode content.
					 *
					 * @since 1.0
					 */
					do_action("lty_shortcode_{$shortcode_name}_content");
					$content = ob_get_contents();
					ob_end_clean();
					break;
			}

			return $content;
		}

		/**
		 * Shortcode dashboard.
		 *
		 * @since 1.0.0
		 * @param array  $atts Shortcode attributes.
		 * @param string $content
		 * @return void
		 */
		public static function shortcode_dashboard( $atts, $content ) {
			if (!is_user_logged_in()) {
				echo wp_kses_post(lty_get_guest_message());
				return;
			}

			LTY_Dashboard::output();
		}

		/**
		 * Shortcode lottery products winners list.
		 * */
		public static function shortcode_lottery_products_winners_list() {
			if ('2' === get_option('lty_settings_single_product_lottery_winner_toggle')) {
				return;
			}

			$winner_ids = lty_get_lottery_winners();
			if (!lty_check_is_array($winner_ids)) {
				return;
			}

			$post_per_page = get_option('lty_settings_winners_list_per_page', 10);
			$current_page = isset($_REQUEST['page_no']) ? wc_clean(wp_unslash(absint($_REQUEST['page_no']))) : '1';
			$offset = ( $post_per_page * $current_page ) - $post_per_page;
			$page_count = ceil(count($winner_ids) / $post_per_page);

			lty_get_template(
					'lottery-product-winners-list-layout.php',
					array(
						'winner_ids' => array_slice($winner_ids, $offset, $post_per_page),
						'offset' => $offset,
						'columns' => lty_get_lottery_shortcode_winner_table_header(),
						'pagination' => lty_prepare_pagination_arguments($current_page, $page_count),
					)
			);
		}

		/**
		 * Shortcode - Display the lottery winners contents by date.
		 *
		 * @since 8.0.0
		 * @param array  $atts
		 * @param string $content
		 * @return string|HTML
		 * */
		public static function shortcode_lottery_winners_by_date( $atts, $content ) {
			$order = isset($atts['order']) ? $atts['order'] : 'DESC';
			$date_filter_number = isset($atts['date_filter_number']) ? $atts['date_filter_number'] : '';
			$date_filter_unit = isset($atts['date_filter_unit']) ? $atts['date_filter_unit'] : '';
			$start_date = lty_prepare_winning_dates_start_date($date_filter_number, $date_filter_unit);
			$lottery_winning_dates = lty_get_lottery_winning_dates($order, $start_date);
			if (!lty_check_is_array($lottery_winning_dates)) {
				return;
			}

			$current_page = 1;
			$per_page = isset($atts['posts_per_page']) ? intval($atts['posts_per_page']) : 99999;
			$paginate = isset($atts['paginate']) ? $atts['paginate'] : false;
			$offset = ( $paginate ) ? ( $per_page * $current_page ) - $per_page : 0;
			$page_count = ceil(count($lottery_winning_dates) / $per_page);

			$table_args = array(
				'lottery_winning_dates' => array_slice($lottery_winning_dates, $offset, $per_page),
				'paginate' => $paginate,
				'per_page' => $per_page,
				'order' => $order,
				'date_filter_number' => $date_filter_number,
				'date_filter_unit' => $date_filter_unit,
				'pagination' => lty_prepare_pagination_arguments($current_page, $page_count),
			);

			lty_get_template('shortcodes/lottery-winners-by-date-layout.php', $table_args);
		}

		/**
		 * Shortcode - Display the lottery instant winners contents by date.
		 *
		 * @since 9.2.0
		 * @param array  $atts Array of attributes.
		 * @param string $content Content.
		 * @return string|HTML
		 * */
		public static function shortcode_lottery_instant_winners_by_date( $atts, $content ) {
			$order = isset($atts['order']) ? $atts['order'] : 'DESC';
			$date_filter_number = isset($atts['date_filter_number']) ? $atts['date_filter_number'] : '';
			$date_filter_unit = isset($atts['date_filter_unit']) ? $atts['date_filter_unit'] : '';
			$start_date = lty_prepare_winning_dates_start_date($date_filter_number, $date_filter_unit);
			$lottery_instant_winning_dates = lty_get_lottery_instant_winning_dates($order, $start_date);
			if (!lty_check_is_array($lottery_instant_winning_dates)) {
				return;
			}

			$current_page = 1;
			$per_page = isset($atts['posts_per_page']) ? intval($atts['posts_per_page']) : 99999;
			$paginate = isset($atts['paginate']) ? $atts['paginate'] : false;
			$offset = ( $paginate ) ? ( $per_page * $current_page ) - $per_page : 0;
			$page_count = ceil(count($lottery_instant_winning_dates) / $per_page);

			$table_args = array(
				'lottery_instant_winning_dates' => array_slice($lottery_instant_winning_dates, $offset, $per_page),
				'paginate' => $paginate,
				'per_page' => $per_page,
				'order' => $order,
				'date_filter_number' => $date_filter_number,
				'date_filter_unit' => $date_filter_unit,
				'pagination' => lty_prepare_pagination_arguments($current_page, $page_count),
			);

			lty_get_template('shortcodes/lottery-instant-winners-by-date-layout.php', $table_args);
		}

		/**
		 * Shortcode My lottery.
		 * */
		public static function shortcode_my_lottery_products( $atts, $content ) {
			if (!is_user_logged_in()) {
				echo wp_kses_post(lty_get_guest_message());
				return;
			}

			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array_merge($attributes, array( 'post__in' => lty_get_my_lotteries() ));
			$args['shortcode_name'] = 'lty_my_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode All lottery.
		 * */
		public static function shortcode_all_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'slug',
						'terms' => array( 'lottery' ), // Set product type as term.
						'operator' => 'IN',
					),
				),
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_all_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode On Going lottery.
		 * */
		public static function shortcode_ongoing_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'slug',
						'terms' => array( 'lottery' ), // Set product type as term.
						'operator' => 'IN',
					),
				),
				'meta_query' => array(
					array(
						'key' => '_lty_lottery_status',
						'value' => 'lty_lottery_started',
						'compare' => '=',
					),
				),
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_ongoing_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode Ending Soon lottery.
		 * */
		public static function shortcode_ending_soon_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'name',
						'terms' => 'lottery',
					),
				),
				'meta_query' => array(
					'relation' => 'AND',
					array(
						'key' => '_lty_end_date_gmt',
						'value' => LTY_Date_Time::get_mysql_date_time_format('now', true),
						'compare' => '>=',
					),
					array(
						'key' => '_lty_lottery_status',
						'value' => 'lty_lottery_started',
						'compare' => '=',
					),
				),
				'meta_key' => '_lty_end_date_gmt',
				'meta_type' => 'DATETIME',
				'orderby' => 'meta_value',
				'order' => 'ASC',
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_ending_soon_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode Future lottery.
		 * */
		public static function shortcode_future_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'name',
						'terms' => 'lottery',
					),
				),
				'meta_query' => array(
					'relation' => 'AND',
					array(
						'key' => '_lty_start_date_gmt',
						'value' => LTY_Date_Time::get_mysql_date_time_format('now', true),
						'compare' => '>=',
					),
					array(
						'key' => '_lty_lottery_status',
						'value' => 'lty_lottery_not_started',
						'compare' => '=',
					),
				),
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_future_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode Featured lottery.
		 * */
		public static function shortcode_featured_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$product_visibility_term_ids = wc_get_product_visibility_term_ids();
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_visibility',
						'field' => 'term_taxonomy_id',
						'terms' => $product_visibility_term_ids['featured'],
					),
					array(
						'taxonomy' => 'product_type',
						'field' => 'name',
						'terms' => 'lottery',
					),
				),
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_featured_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode Closed lottery.
		 * */
		public static function shortcode_closed_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'name',
						'terms' => 'lottery',
					),
				),
				'meta_query' => array(
					array(
						'key' => '_lty_lottery_status',
						'value' => 'lty_lottery_closed',
						'compare' => '=',
					),
				),
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_closed_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode finished lottery products.
		 * */
		public static function shortcode_finished_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'name',
						'terms' => 'lottery',
					),
				),
				'meta_query' => array(
					array(
						'key' => '_lty_lottery_status',
						'value' => 'lty_lottery_finished',
						'compare' => '=',
					),
				),
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_finished_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode Random lottery.
		 * */
		public static function shortcode_random_lottery_products( $atts, $content ) {
			$attributes = array_filter(self::shortcode_attributes($atts));
			$args = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'product_type',
						'field' => 'name',
						'terms' => 'lottery',
					),
				),
				'orderby' => 'rand',
			);

			$args = array_merge($attributes, $args);
			$args['shortcode_name'] = 'lty_random_lottery';

			return self::render_lottery_products($args);
		}

		/**
		 * Shortcode Attributes.
		 *
		 * @since 1.0.0
		 * @param array $atts Shortcode Attributes.
		 * @return array
		 */
		public static function shortcode_attributes( $atts ) {
			$atts = array_filter(
					shortcode_atts(
							array(
								'posts_per_page' => '',
								'order' => '',
								'orderby' => '',
								'paginate' => '',
								'category' => '',
							),
							$atts,
							'lty_lottery_products'
					)
			);

			$attributes = array(
				'posts_per_page' => isset($atts['posts_per_page']) ? $atts['posts_per_page'] : '-1',
				'order' => isset($atts['order']) ? $atts['order'] : 'DESC',
				'orderby' => isset($atts['orderby']) ? $atts['orderby'] : 'date',
				'paginate' => isset($atts['paginate']) ? $atts['paginate'] : false,
				'category' => isset($atts['category']) ? $atts['category'] : '',
			);

			switch ($attributes['orderby']) {
				case 'start_date':
					$attributes['meta_key'] = '_lty_start_date_gmt';
					$attributes['orderby'] = 'meta_value';
					break;
				case 'end_date':
					$attributes['meta_key'] = '_lty_end_date_gmt';
					$attributes['orderby'] = 'meta_value';
					break;
				case 'finished_date':
					$attributes['meta_key'] = '_lty_finished_date_gmt';
					$attributes['orderby'] = 'meta_value';
					break;
				case 'closed_date':
					$attributes['meta_key'] = '_lty_closed_date_gmt';
					$attributes['orderby'] = 'meta_value';
					break;
				case 'failed_date':
					$attributes['meta_key'] = '_lty_failed_date_gmt';
					$attributes['orderby'] = 'meta_value';
					break;
				case 'remaining_ticket_count':
					$attributes['meta_key']  = '_stock';
					$attributes['orderby']   = 'meta_value';
					$attributes['meta_type'] = 'NUMERIC';
					break;
			}

			return $attributes;
		}

		/**
		 * Render Lottery Products.
		 *
		 * @since 1.0.0
		 * @param array $args Query arguments.
		 * @return string|HTML
		 */
		public static function render_lottery_products( $args = array() ) {
			$object = new LTY_Shortcode_Products($args);

			return $object->get_content();
		}
	}

	LTY_Shortcodes::init();
}
