<?php

/**
 * Shortcodes - Product page.
 *
 * @since 10.2.0
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('LTY_Product_Page_Shortcodes')) {

	/**
	 * Class.
	 *
	 * @since 10.2.0
	 */
	class LTY_Product_Page_Shortcodes {

		/**
		 * Class.
		 *
		 * @since 10.2.0
		 * */
		public static function init() {
			/**
			 * This hook is used to alter the load product page shortcodes.
			 *
			 * @since 10.2.0
			 */
			$shortcodes = apply_filters(
					'lty_load_product_page_shortcodes',
					array(
						'lty_lottery_status',
						'lty_lottery_start_date',
						'lty_lottery_end_date',
						'lty_lottery_date_notice',
						'lty_lottery_count_down_timer',
						'lty_lottery_minimum_tickets',
						'lty_lottery_maximum_tickets',
						'lty_lottery_maximum_tickets_per_user',
						'lty_lottery_minimum_tickets_per_user',
						'lty_lottery_maximum_tickets_per_user_notice',
						'lty_lottery_progress_bar',
						'lty_lottery_tickets_sold_percentage',
						'lty_lottery_tickets_sold_count',
						'lty_lottery_winning_item',
						'lty_lottery_question_answer',
						'lty_lottery_quantity_selector',
						'lty_lottery_participate_button',
						'lty_lottery_predefined_button_url',
						'lty_lottery_predefined_button_amount',
						'lty_lottery_predefined_button_total_amount',
						'lty_lottery_predefined_button_discount',
						'lty_lottery_predefined_button_tickets_quantity',
						'lty_instant_win_prizes',
						'lty_user_chooses_ticket',
						'lty_lucky_dip',
						'lty_lottery_details_tab',
					)
			);

			foreach ($shortcodes as $shortcode_name) {
				add_shortcode($shortcode_name, array( __CLASS__, 'process_shortcode' ));
			}
		}

		/**
		 * Process Shortcode.
		 *
		 * @since 10.2.0
		 * @param array  $atts Shortcode attributes.
		 * @param string $content
		 * @param string $tag
		 * @return string
		 */
		public static function process_shortcode( $atts, $content, $tag ) {
			$shortcode_name = str_replace('lty_', '', $tag);
			$function = 'shortcode_' . $shortcode_name;

			switch ($shortcode_name) {
				case 'lottery_details_tab':
				case 'lottery_question_answer':
				case 'lottery_quantity_selector':
					ob_start();
					self::$function($atts, $content);
					$content = ob_get_contents();
					ob_end_clean();
					break;

				case 'lottery_status':
				case 'lottery_start_date':
				case 'lottery_end_date':
				case 'lottery_date_notice':
				case 'lottery_count_down_timer':
				case 'lottery_minimum_tickets':
				case 'lottery_maximum_tickets':
				case 'lottery_maximum_tickets_per_user':
				case 'lottery_minimum_tickets_per_user':
				case 'lottery_maximum_tickets_per_user_notice':
				case 'lottery_progress_bar':
				case 'lottery_tickets_sold_percentage':
				case 'lottery_tickets_sold_count':
				case 'lottery_winning_item':
				case 'lottery_participate_button':
				case 'lottery_predefined_button_url':
				case 'lottery_predefined_button_amount':
				case 'lottery_predefined_button_total_amount':
				case 'lottery_predefined_button_discount':
				case 'lottery_predefined_button_tickets_quantity':
				case 'instant_win_prizes':
				case 'user_chooses_ticket':
				case 'lucky_dip':
					$content = self::$function($atts, $content);
					break;
				default:
					ob_start();

					/**
					 * This hook is used to display the product page shortcode content.
					 *
					 * @since 10.2.0
					 */
					do_action("lty_product_page_shortcode_{$shortcode_name}_content");
					$content = ob_get_contents();
					ob_end_clean();
					break;
			}

			return $content;
		}

		/**
		 * Shortcode lottery details tab.
		 * */
		public static function shortcode_lottery_details_tab( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			return lty_get_template('single-product/tabs/ticket-logs-layout.php', lty_prepare_ticket_logs_template_arguments($lottery_product));
		}

		/**
		 * Shortcode lottery status.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 */
		public static function shortcode_lottery_status( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			return lty_get_template_html('single-product/ticket-status.php', array( 'product' => $lottery_product ));
		}

		/**
		 * Shortcode lottery start date.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string
		 */
		public static function shortcode_lottery_start_date( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product) || !$lottery_product->get_lty_start_date()) {
				return;
			}

			$display_timezone = isset($atts['display_timezone']) ? ( 'false' === $atts['display_timezone'] ? false : true ) : true;

			return LTY_Date_Time::get_wp_format_datetime($lottery_product->get_lty_start_date(), false, false, false, ' ', $display_timezone);
		}

		/**
		 * Shortcode lottery end date.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string
		 */
		public static function shortcode_lottery_end_date( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product) || !$lottery_product->get_lty_end_date()) {
				return;
			}

			$display_timezone = isset($atts['display_timezone']) ? ( 'false' == $atts['display_timezone'] ? false : true ) : true;

			return LTY_Date_Time::get_wp_format_datetime($lottery_product->get_lty_end_date(), false, false, false, ' ', $display_timezone);
		}

		/**
		 * Shortcode lottery date notice.
		 *
		 * @since 9.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 */
		public static function shortcode_lottery_date_notice( $atts ) {
			$product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($product)) {
				return;
			}

			if (!$product->is_valid_to_display_countdown_timer_in_product_page('shortcode')) {
				return;
			}

			return lty_get_template_html('single-product/date-notice.php', array( 'product' => $product ));
		}

		/**
		 * Shortcode lottery count down timer.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 */
		public static function shortcode_lottery_count_down_timer( $atts ) {
			$product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($product)) {
				return;
			}

			if (!$product->is_valid_to_display_countdown_timer_in_product_page('shortcode')) {
				return;
			}

			return lty_get_template_html('single-product/date-ranges.php', array( 'product' => $product ));
		}

		/**
		 * Shortcode minimum tickets.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return int
		 */
		public static function shortcode_lottery_minimum_tickets( $atts ) {
			$product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($product)) {
				return;
			}

			return absint($product->get_lty_minimum_tickets());
		}

		/**
		 * Shortcode maximum tickets.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return int
		 */
		public static function shortcode_lottery_maximum_tickets( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			return absint($lottery_product->get_lty_maximum_tickets());
		}

		/**
		 * Shortcode maximum tickets per user.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return int
		 */
		public static function shortcode_lottery_maximum_tickets_per_user( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			if ('3' == get_option('lty_settings_guest_user_participate_type')) {
				return;
			}

			return absint($lottery_product->get_lty_user_maximum_tickets());
		}

		/**
		 * Shortcode minimum tickets per user.
		 *
		 * @return int.
		 * */
		public static function shortcode_lottery_minimum_tickets_per_user( $atts ) {
			if ('3' == get_option('lty_settings_guest_user_participate_type')) {
				return;
			}

			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			return absint($lottery_product->get_lty_user_minimum_tickets());
		}

		/**
		 * Shortcode lottery maximum tickets per user notice.
		 *
		 * @return string.
		 * */
		public static function shortcode_lottery_maximum_tickets_per_user_notice( $atts ) {
			if (!wp_get_current_user()->exists()) {
				return;
			}

			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			// return if the user purchase limit is not reached.
			if (!$lottery_product->user_purchase_limit_exists() || $lottery_product->is_closed()) {
				return;
			}

			$message = get_option('lty_settings_maximum_tickets_purchase_limit_error_message', 'You have reached the Maximum ticket(s) count {maximum_tickets_count} for this lottery. Hence you cannot purchase new lottery tickets.');
			$message = str_replace('{maximum_tickets_count}', '<b>' . $lottery_product->get_lty_user_maximum_tickets() . '</b>', $message);

			return $message;
		}

		/**
		 * Shortcode lottery progress bar.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 */
		public static function shortcode_lottery_progress_bar( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product) || $lottery_product->is_closed()) {
				return;
			}

			$args = array(
				'product' => $lottery_product,
				'progress_bar_percentage' => lty_get_product_page_progress_bar_percentage($lottery_product),
			);

			return lty_get_template_html('single-product/progress-bar.php', $args);
		}

		/**
		 * Shortcode lottery tickets sold percentage.
		 *
		 * @return string.
		 * */
		public static function shortcode_lottery_tickets_sold_percentage( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product) || $lottery_product->is_closed()) {
				return;
			}

			$sold_percentage = isset($atts['decimal_count']) && is_numeric($atts['decimal_count']) ? round(lty_get_progress_bar_percentage($lottery_product), (int) $atts['decimal_count']) : wc_format_decimal(lty_get_progress_bar_percentage($lottery_product), '');

			return $sold_percentage . '%';
		}

		/**
		 * Shortcode lottery tickets sold count.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return int
		 */
		public static function shortcode_lottery_tickets_sold_count( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product) || $lottery_product->is_closed()) {
				return;
			}

			return absint($lottery_product->get_lottery_ticket_count());
		}

		/**
		 * Shortcode lottery winning item.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 * */
		public static function shortcode_lottery_winning_item( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			return lty_get_lottery_gift_products(false, $lottery_product, true);
		}

		/**
		 * Shortcode lottery question answer.
		 *
		 * @return mixed
		 * */
		public static function shortcode_lottery_question_answer( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			return LTY_Lottery_Single_Product_Templates::render_question_answer_template($lottery_product);
		}

		/**
		 * Shortcode lottery quantity selector.
		 *
		 * @since 7.0.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 * */
		public static function shortcode_lottery_quantity_selector( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			if (!$lottery_product->is_purchasable() || !$lottery_product->is_in_stock() || !$lottery_product->is_started() || $lottery_product->is_closed() || $lottery_product->user_purchase_limit_exists()) {
				return;
			}

			if (( $lottery_product->can_display_predefined_buttons() && !$lottery_product->can_display_predefined_with_quantity_selector() ) || $lottery_product->is_manual_ticket()) {
				return;
			}

			$customer_id = lty_get_current_user_cart_session_value();
			// Return if the user selected incorrect answer.
			if ($lottery_product->validate_user_incorrect_answer($customer_id) || $lottery_product->is_customer_question_answer_time_limit_exists($customer_id)) {
				return;
			}

			lty_render_quantity_field($lottery_product);
		}

		/**
		 * Shortcode - lottery participate button.
		 *
		 * @since 7.0.0
		 * @param array $atts Shortcode attributes.
		 * @return string|HTML
		 * */
		public static function shortcode_lottery_participate_button( $atts ) {
			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return;
			}

			if (!$lottery_product->is_purchasable() || !$lottery_product->is_in_stock() || !$lottery_product->is_started() || $lottery_product->is_closed() || $lottery_product->user_purchase_limit_exists()) {
				return;
			}

			$customer_id = lty_get_current_user_cart_session_value();
			// Return if the user selected incorrect answer.
			if ($lottery_product->validate_user_incorrect_answer($customer_id) || $lottery_product->is_customer_question_answer_time_limit_exists($customer_id)) {
				return;
			}

			return lty_get_template_html('shortcodes/participate-button.php', array( 'product' => $lottery_product ));
		}

		/**
		 * Shortcode lottery predefined button URL.
		 *
		 * @since 7.3.0
		 * @param array $atts Shortcode attributes.
		 * @return string|URL
		 */
		public static function shortcode_lottery_predefined_button_url( $atts ) {
			$lottery_product = self::is_valid_lottery_predefined_button($atts);
			if (!is_object($lottery_product)) {
				return;
			}

			return add_query_arg(
					array(
						'lty_product_id' => $lottery_product->get_id(),
						'quantity' => $lottery_product->get_predefined_buttons_ticket_quantity($atts['button_key']),
						'button_key' => $atts['button_key'],
					),
					$lottery_product->get_permalink()
			);
		}

		/**
		 * Shortcode lottery predefined button amount.
		 *
		 * @since 7.3.0
		 * @param array $atts Shortcode attributes.
		 * @return float
		 */
		public static function shortcode_lottery_predefined_button_amount( $atts ) {
			$lottery_product = self::is_valid_lottery_predefined_button($atts);
			if (!is_object($lottery_product)) {
				return;
			}

			return $lottery_product->get_predefined_buttons_per_ticket_amount($atts['button_key']);
		}

		/**
		 * Shortcode lottery predefined button total amount.
		 *
		 * @since 7.3.0
		 * @param array $atts Shortcode attributes.
		 * @return float
		 */
		public static function shortcode_lottery_predefined_button_total_amount( $atts ) {
			$lottery_product = self::is_valid_lottery_predefined_button($atts);
			if (!is_object($lottery_product)) {
				return;
			}

			return $lottery_product->get_predefined_buttons_total_ticket_amount($atts['button_key']);
		}

		/**
		 * Shortcode lottery predefined button discount.
		 *
		 * @since 7.3.0
		 * @param array $atts Shortcode attributes.
		 * @return float
		 */
		public static function shortcode_lottery_predefined_button_discount( $atts ) {
			$lottery_product = self::is_valid_lottery_predefined_button($atts);
			if (!is_object($lottery_product)) {
				return;
			}

			return $lottery_product->get_predefined_buttons_discount_amount($atts['button_key']);
		}

		/**
		 * Shortcode lottery predefined button ticket quantity.
		 *
		 * @since 7.3.0
		 * @param array $atts Shortcode attributes.
		 * @return int
		 */
		public static function shortcode_lottery_predefined_button_tickets_quantity( $atts ) {
			$lottery_product = self::is_valid_lottery_predefined_button($atts);
			if (!is_object($lottery_product)) {
				return;
			}

			return $lottery_product->get_predefined_buttons_ticket_quantity($atts['button_key']);
		}

		/**
		 * Shortcode - Instant Winners Prizes.
		 *
		 * @since 8.0.0
		 * @return HTML
		 * */
		public static function shortcode_instant_win_prizes( $atts, $content ) {
			$product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($product) || !$product->is_instant_winner()) {
				return;
			}

			if (!lty_check_is_array($product->get_current_instant_winner_log_ids())) {
				return;
			}

			return lty_get_template_html('single-product/tabs/instant-winners-logs-layout.php', lty_prepare_instant_winner_logs_arguments($product));
		}

		/**
		 * Shortcode - User chooses the ticket
		 *
		 * @since 8.5.0
		 * @param array  $atts
		 * @param string $content
		 * @return string|HTML
		 * */
		public static function shortcode_user_chooses_ticket( $atts, $content ) {
			$product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($product) || !$product->is_in_stock()) {
				return;
			}

			if (!$product->is_started() || $product->is_closed() || !$product->is_manual_ticket() || $product->user_purchase_limit_exists()) {
				return;
			}

			return lty_get_template_html('single-product/ticket-summary.php', array( 'product' => $product ));
		}

		/**
		 * Shortcode - Lucky dip
		 *
		 * @since 8.5.0
		 * @param array  $atts
		 * @param string $content
		 * @return string|HTML
		 * */
		public static function shortcode_lucky_dip( $atts, $content ) {
			$product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($product) || !$product->is_in_stock() || 'yes' !== $product->get_lty_lucky_dip()) {
				return;
			}

			if (!$product->is_started() || $product->is_closed() || !$product->is_manual_ticket() || $product->user_purchase_limit_exists()) {
				return;
			}

			return lty_get_template_html('single-product/ticket-lucky-dip.php', array( 'product' => $product ));
		}

		/**
		 * Shortcode lottery predefined button discount.
		 *
		 * @since 7.3.0
		 * @param array $atts Shortcode attributes.
		 * @return string|object
		 */
		public static function is_valid_lottery_predefined_button( $atts ) {
			if ('' == $atts['button_key']) {
				return false;
			}

			$lottery_product = self::get_product_based_on_shortcode_attributes($atts);
			if (!lty_is_lottery_product($lottery_product)) {
				return false;
			}

			if (!$lottery_product->is_purchasable() || !$lottery_product->is_in_stock() || !$lottery_product->is_started() || $lottery_product->is_closed() || $lottery_product->user_purchase_limit_exists()) {
				return false;
			}

			$customer_id = lty_get_current_user_cart_session_value();
			// Return if the user selected incorrect answer.
			if ($lottery_product->validate_user_incorrect_answer($customer_id) || $lottery_product->is_customer_question_answer_time_limit_exists($customer_id)) {
				return false;
			}

			if ($lottery_product->is_manual_ticket() || !$lottery_product->is_predefined_button_enabled()) {
				return false;
			}

			if (!array_key_exists($atts['button_key'], $lottery_product->get_predefined_buttons_rule())) {
				return false;
			}

			return $lottery_product;
		}

		/**
		 * Get product based on shortcode attributes.
		 *
		 * @since 10.2.0
		 * @param array $atts Shortcode attributes.
		 * @return object Product object.
		 */
		public static function get_product_based_on_shortcode_attributes( $atts ) {
			global $product;
			$product_id = isset($atts['product_id']) ? absint($atts['product_id']) : '';

			return !empty($product_id) ? wc_get_product($product_id) : $product;
		}
	}

	LTY_Product_Page_Shortcodes::init();
}
